##########################################################################
# A Paraview script to render an animation (movie file) from a set
# of VTK .vtm files which must be in a 'VTK' sub-directory in the current
# direcotry. These are generated by running 'foamToVTK' after an OpenFOAM
# 'redistributePar -reconstruct' step.
#
# Required args: name-of-anim-file.ogv
#
# NOTE: only .ogv or .avi files are supported!
#
# The format of the output file will be determined by the file-extension.
#
# The initial .py file was created by starting a 'Trace' in paraview
# to record the steps taken in the GUI. Stop the Trace once you've
# done everything you want to record. This then gives a good starting
# point for a script.
#
# If run in pvbatch from the paraview GUI version, it will still try
# to pop-up a render window. If run in pvbatch from the paraview OSMesa
# version it will use off-screen rendering, which is what we want for
# a batch job.
#
# Hand-rolled code will glob the .vtm files in a VTK/ subdir, which the
# foamToVTK util generates. Some of the paraview commands to render
# legends have been removed because we switch to rendering the
# alpha.water data and yet the orginal vtkBlockColor colourbar legend
# was still visible in the OSMesa version.
#
# george.leaver@manchester.ac.uk, RI Team, Mar 2022
##########################################################################
import sys
import os.path
import glob
import re
from os import stat as stat
from paraview.simple import *

if len(sys.argv) != 2:
   sys.exit( "ERROR: Please supply the name of the movie file (e.g., anim.ogv)" )

filename=sys.argv[1]
if os.path.splitext( filename )[-1] not in [ '.ogv', '.avi' ]:
   sys.exit("ERROR: movie file must be a .avi or .ogv file.")

#filename='foam-anim.ogv'

#### disable automatic camera reset on 'Show'
paraview.simple._DisableFirstRenderCameraReset()

# create a new 'XML MultiBlock Data Reader'
# GWL: Paraview will hard-code the .vtm files so instead we glob them.
# foamexample_ = XMLMultiBlockDataReader(FileName=['VTK/foam-example_0.vtm', 'VTK/foam-example_14.vtm', ..., 'VTK/foam-example_1581.vtm'])
#
# The glob() func doesn't sort on the number in the filename as you'd like (is gives xyz_1, xyz_10, xyz_100, xyz_2, ...)
# So we need to supply a function to sorted() that extracts the number so it can sort just on that.

onlyVTMFiles_sorted = sorted( glob.glob( 'VTK/' + '*.vtm'), key=lambda x:float(re.findall("(\d+)",x)[0]))
foamexample_ = XMLMultiBlockDataReader(FileName=onlyVTMFiles_sorted)

# get animation scene
animationScene1 = GetAnimationScene()

# update animation scene based on data timesteps
animationScene1.UpdateAnimationUsingDataTimeSteps()

# get active view
renderView1 = GetActiveViewOrCreate('RenderView')
# uncomment following to set a specific view size
#renderView1.ViewSize = [642, 447]

# GWL: Changed to the animation generator's preferred size. It tell you this in the GUI.
renderView1.ViewSize = [624, 456]

# show data in view
foamexample_Display = Show(foamexample_, renderView1)

# reset view to fit data
renderView1.ResetCamera()

# set scalar coloring
ColorBy(foamexample_Display, ('POINTS', 'alpha.water'))

# rescale color and/or opacity maps used to include current data range
foamexample_Display.RescaleTransferFunctionToDataRange(True)

# show color bar/color legend
foamexample_Display.SetScalarBarVisibility(renderView1, True)

# get color transfer function/color map for 'alphawater'
alphawaterLUT = GetColorTransferFunction('alphawater')

# get opacity transfer function/opacity map for 'alphawater'
alphawaterPWF = GetOpacityTransferFunction('alphawater')

# current camera placement for renderView1
renderView1.CameraPosition = [0.292, 0.292, 1.603]
renderView1.CameraFocalPoint = [0.292, 0.292, 0.0073]
renderView1.CameraParallelScale = 0.4130

# save animation images/movie
# GWL: This is deprecated in favour of SaveAnimation for PV 5.6 and beyond, but still works perfectly well.
WriteAnimation(filename, Magnification=1, FrameRate=15.0, Compression=True)


try: status=stat(filename).st_size>0
except: sys.exit(filename+" something went wrong saving the animation")

if (status==True):
   print(filename,"written successfully")
else:
   print(filename,"is empty - ERROR!")

